(() => {
  const $ = (s) => document.querySelector(s);
  const $$ = (s) => Array.from(document.querySelectorAll(s));

  const base = document.querySelector('meta[name="app-base"]')?.content || '/app';
  const csrf = document.querySelector('meta[name="csrf-token"]')?.content || '';

  function api(path){ return base + '/api' + path; }

  const isCreate = !!$('#generateBtn');
  if(!isCreate) return;

  const viewAds = $('#viewAds');
  const viewCopy = $('#viewCopy');
  const viewTrash = $('#viewTrash');

  const adsGrid = $('#adsGrid');
  const trashGrid = $('#trashGrid');

  function setView(which){
    $('#tabAds')?.classList.toggle('active', which==='ads');
    $('#tabCopy')?.classList.toggle('active', which==='copy');
    $('#tabTrash')?.classList.toggle('active', which==='trash');

    $('#tabAds')?.setAttribute('aria-selected', which==='ads'?'true':'false');
    $('#tabCopy')?.setAttribute('aria-selected', which==='copy'?'true':'false');
    $('#tabTrash')?.setAttribute('aria-selected', which==='trash'?'true':'false');

    if(viewAds) viewAds.style.display = (which==='ads') ? '' : 'none';
    if(viewCopy) viewCopy.style.display = (which==='copy') ? '' : 'none';
    if(viewTrash) viewTrash.style.display = (which==='trash') ? '' : 'none';

    const title = which==='ads' ? 'Generated Ads' : (which==='copy' ? 'Generated Copy' : 'Trash');
    $('#viewTitle') && ($('#viewTitle').textContent = title);

    if(which === 'trash') loadTrash();
  }

  $('#tabAds')?.addEventListener('click', ()=>setView('ads'));
  $('#tabCopy')?.addEventListener('click', ()=>setView('copy'));
  $('#tabTrash')?.addEventListener('click', ()=>setView('trash'));

  // Size tabs (UI only)
  $$('.tab').forEach(tab=>{
    tab.addEventListener('click', ()=>{
      $$('.tab').forEach(t=>t.classList.remove('active'));
      tab.classList.add('active');
    });
  });

  // Switches
  $$('.switch').forEach(sw=>{
    sw.addEventListener('click', ()=>{
      sw.classList.toggle('on');
      sw.setAttribute('aria-checked', sw.classList.contains('on') ? 'true' : 'false');
      calcCost();
    });
  });

  // Drawer
  const overlay = $('#drawerOverlay');
  const openDrawer = () => { overlay?.classList.add('open'); overlay?.setAttribute('aria-hidden','false'); calcCost(); };
  const closeDrawer = () => { overlay?.classList.remove('open'); overlay?.setAttribute('aria-hidden','true'); calcCost(); };

  $('#openAdvanced')?.addEventListener('click', openDrawer);
  $('#openAdvancedRight')?.addEventListener('click', openDrawer);
  $('#closeDrawer')?.addEventListener('click', closeDrawer);
  $('#saveAdvanced')?.addEventListener('click', closeDrawer);
  overlay?.addEventListener('click', (e)=>{ if(e.target === overlay) closeDrawer(); });

  // Toast
  const toast = $('#toast');
  const toastMsg = $('#toastMsg');
  const undoBtn = $('#undoBtn');
  let undoAction = null;

  function showToast(message, undoFn=null){
    if(!toast || !toastMsg) return;
    toastMsg.textContent = message;
    undoAction = undoFn;
    if(undoBtn){
      undoBtn.style.display = undoFn ? '' : 'none';
    }
    toast.classList.add('show');
    if(!undoFn){
      setTimeout(()=>toast.classList.remove('show'), 2400);
    }
  }
  $('#dismissToast')?.addEventListener('click', ()=>toast?.classList.remove('show'));
  undoBtn?.addEventListener('click', ()=>{
    if(undoAction) undoAction();
    toast?.classList.remove('show');
    undoAction = null;
  });

  // Cost hint (client-side estimate)
  function calcCost(){
    const vars = parseInt($('#variations')?.value || '4', 10);
    const q = ($('#quality')?.value || 'medium').toLowerCase();
    const mult = (q === 'low') ? 0.75 : (q === 'medium') ? 1 : (q === 'high') ? 1.6 : 1.1;
    let cost = Math.max(1, Math.round(vars * mult));
    if($('#copySwitch')?.classList.contains('on')) cost += 1;
    $('#costHint') && ($('#costHint').textContent = `Generates ${vars} ads · Costs ${cost} credits`);
  }
  ['#variations','#quality'].forEach(sel => $(sel)?.addEventListener('change', calcCost));
  calcCost();

  // Copy hint
  function updateCopyHint(){
    const obj = $('#objective')?.value || 'Leads';
    $('#copyHint') && ($('#copyHint').textContent = `Objective: ${obj}`);
  }
  $('#objective')?.addEventListener('change', updateCopyHint);
  updateCopyHint();

  // Render assets
  function makeAdCard(asset){
    const id = asset.id;
    const badge = asset.variant_label || ('Asset #' + id);
    const imgUrl = asset.public_url;

    const card = document.createElement('div');
    card.className = 'adCard';
    card.dataset.assetId = String(id);

    card.innerHTML = `
      <button class="trashBtn" title="Move to Trash" aria-label="Move to Trash" data-trash="${id}">
        <span class="tip">Move to Trash</span>
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
          <path d="M3 6h18" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
          <path d="M8 6V4h8v2" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
          <path d="M6 6l1 14h10l1-14" stroke="#fff" stroke-width="2" stroke-linejoin="round"/>
          <path d="M10 11v6M14 11v6" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
        </svg>
      </button>

      <div class="adPreview">
        <div class="badge">${escapeHtml(badge)}</div>
        <img alt="Generated ad" src="${escapeAttr(imgUrl)}" style="width:100%;height:100%;object-fit:cover;" />
      </div>

      <div class="adActions">
        <div class="iconRow">
          <button class="iconBtn" title="Download" aria-label="Download" data-download="${escapeAttr(imgUrl)}">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
              <path d="M12 3v10" stroke="#b0b0b0" stroke-width="1.8" stroke-linecap="round"/>
              <path d="M8 11l4 4 4-4" stroke="#b0b0b0" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round"/>
              <path d="M4 20h16" stroke="#b0b0b0" stroke-width="1.8" stroke-linecap="round"/>
            </svg>
          </button>
        </div>
        <button class="savePill" title="Save to My Ads" data-save="${id}">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
            <path d="M6 4h12v16l-6-3-6 3V4z" stroke="#fff" stroke-width="1.6" stroke-linejoin="round"/>
          </svg>
          Save
        </button>
      </div>
    `;
    return card;
  }

  function makeTrashCard(asset){
    const id = asset.id;
    const badge = asset.variant_label || ('Asset #' + id);
    const imgUrl = asset.public_url;

    const card = document.createElement('div');
    card.className = 'adCard';
    card.dataset.assetId = String(id);

    card.innerHTML = `
      <button class="trashBtn" title="Restore" aria-label="Restore" data-restore="${id}">
        <span class="tip">Restore</span>
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
          <path d="M12 5v4" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
          <path d="M12 9h7" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
          <path d="M21 12a9 9 0 1 1-3-6.7" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
        </svg>
      </button>

      <div class="adPreview">
        <div class="badge">${escapeHtml(badge)}</div>
        <img alt="Trashed ad" src="${escapeAttr(imgUrl)}" style="width:100%;height:100%;object-fit:cover;" />
      </div>

      <div class="adActions">
        <div class="iconRow">
          <button class="iconBtn" title="Download" aria-label="Download" data-download="${escapeAttr(imgUrl)}">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
              <path d="M12 3v10" stroke="#b0b0b0" stroke-width="1.8" stroke-linecap="round"/>
              <path d="M8 11l4 4 4-4" stroke="#b0b0b0" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round"/>
              <path d="M4 20h16" stroke="#b0b0b0" stroke-width="1.8" stroke-linecap="round"/>
            </svg>
          </button>
        </div>
        <button class="savePill" title="Restore" data-restore2="${id}">Restore</button>
      </div>
    `;
    return card;
  }

  function setCounts(){
    $('#countAds') && ($('#countAds').textContent = String(adsGrid?.children.length || 0));
    $('#countTrash') && ($('#countTrash').textContent = String(trashGrid?.children.length || 0));
  }

  // Copy rendering
  function renderCopy(list){
    const wrap = $('#copyList');
    if(!wrap) return;
    wrap.innerHTML = '';
    if(!list || !list.length){
      wrap.innerHTML = `
        <div class="copyCard">
          <div class="copyMeta">
            <div class="ttl">No copy yet</div>
            <span class="pillMini">Turn on “Generate Copy” then click Generate</span>
          </div>
          <div class="copyText">Copy variants will appear here.</div>
          <div class="copyActions">
            <span class="pillMini">Tip: Generate 3–5 variants per offer.</span>
            <button class="copyBtn" disabled>Copy Text</button>
          </div>
        </div>`;
      $('#countCopy') && ($('#countCopy').textContent = '0');
      return;
    }
    $('#countCopy') && ($('#countCopy').textContent = String(list.length));
    list.forEach((c, idx)=>{
      const card = document.createElement('div');
      card.className = 'copyCard';
      card.innerHTML = `
        <div class="copyMeta">
          <div class="ttl">Variant ${idx+1}</div>
          <span class="pillMini">${escapeHtml(c.copy_type || 'Copy')}</span>
        </div>
        <div class="copyText" id="copyText_${idx}">${escapeHtml(c.full_text || '')}</div>
        <div class="copyActions">
          <span class="pillMini">${escapeHtml($('#objective')?.value || '')}</span>
          <button class="copyBtn" data-copy="copyText_${idx}">Copy Text</button>
        </div>
      `;
      wrap.appendChild(card);
    });
  }

  // HTML escape helpers
  function escapeHtml(str){
    return String(str).replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[s]));
  }
  function escapeAttr(str){
    return escapeHtml(str).replace(/\s+/g,' ');
  }

  // API calls
async function postJson(url, body){
  const res = await fetch(url, {
    method:'POST',
    headers: {
      'Content-Type':'application/json',
      'X-CSRF-Token': csrf
    },
    body: JSON.stringify(body || {})
  });

  const raw = await res.text();
  let data;
  try {
    data = JSON.parse(raw);
  } catch (e) {
    // Show the first part of the raw response (usually an HTML/PHP error)
    throw new Error("Bad JSON from server. First 300 chars: " + raw.slice(0, 300));
  }

  if(!res.ok || !data.ok){
    throw new Error(data.error || ('HTTP ' + res.status));
  }
  return data;
}


  async function generate(){
    const product = $('#product')?.value || '';
    const offer = $('#offer')?.value || '';
    const objective = $('#objective')?.value || 'Leads';
    const placement = $('#placement')?.value || 'Facebook Feed';
    const cta = $('#cta')?.value || '';
    const generate_copy = $('#copySwitch')?.classList.contains('on') || false;

    const style_name = document.querySelector('.styleItem.active')?.innerText.trim() || 'Flash Sale';
    const variant = document.querySelector('.variantBtn.active')?.innerText.trim() || 'V2';

    const gen_size = $('#genSize')?.value || 'Square (1024×1024)';
    const quality = ($('#quality')?.value || 'medium').toLowerCase();
    const variations = parseInt($('#variations')?.value || '4', 10);

    const copy_type = ($('#copyType')?.value || 'Facebook Ad Copy (Primary + Headline)').includes('Post') ? 'Facebook Post Copy' : 'Facebook Ad Copy';
    const copy_variants = parseInt($('#copyVariants')?.value || '3', 10);
    const copy_tone = $('#copyTone')?.value || 'Direct response';

    const payload = {
      product, offer, objective, placement, cta,
      style_name, variant,
      gen_size, quality, variations,
      generate_copy,
      copy_type,
      copy_variants,
      copy_tone
    };

    $('#generateBtn').disabled = true;
    $('#generateBtn').textContent = 'Generating...';

    try{
      const data = await postJson(api('/generate.php'), payload);

      // Render ads
      adsGrid.innerHTML = '';
      (data.assets || []).forEach(a => adsGrid.appendChild(makeAdCard(a)));
      setCounts();

      // Render copy
      renderCopy(data.copy || []);

      showToast(`Generated ${data.assets?.length || 0} ads`, null);
      setView('ads');
    }catch(e){
      showToast('Error: ' + e.message, null);
    }finally{
      $('#generateBtn').disabled = false;
      $('#generateBtn').textContent = 'Generate';
    }
  }

  // Trash handlers
  document.addEventListener('click', async (e)=>{
    const t = e.target.closest('[data-trash]');
    if(t){
      const assetId = parseInt(t.getAttribute('data-trash'), 10);
      try{
        await postJson(api('/trash_asset.php'), { asset_id: assetId });
        const card = t.closest('.adCard');
        card && card.remove();
        setCounts();
        showToast('Moved to Trash', async ()=> {
          await postJson(api('/restore_asset.php'), { asset_id: assetId });
          await refreshLatest();
        });
      }catch(err){
        showToast('Error: ' + err.message, null);
      }
      return;
    }

    const r = e.target.closest('[data-restore], [data-restore2]');
    if(r){
      const assetId = parseInt(r.getAttribute('data-restore') || r.getAttribute('data-restore2'), 10);
      try{
        await postJson(api('/restore_asset.php'), { asset_id: assetId });
        await loadTrash();
        await refreshLatest();
        showToast('Restored', null);
      }catch(err){
        showToast('Error: ' + err.message, null);
      }
      return;
    }

    const d = e.target.closest('[data-download]');
    if(d){
      const url = d.getAttribute('data-download');
      window.open(url, '_blank');
      return;
    }

    const s = e.target.closest('[data-save]');
    if(s){
      const assetId = parseInt(s.getAttribute('data-save'), 10);
      try{
        await postJson(api('/save_ad.php'), { asset_id: assetId });
        showToast('Saved to My Ads', null);
      }catch(err){
        showToast('Error: ' + err.message, null);
      }
      return;
    }

    const copyBtn = e.target.closest('[data-copy]');
    if(copyBtn){
      const id = copyBtn.getAttribute('data-copy');
      const txt = document.getElementById(id)?.innerText || '';
      try{ await navigator.clipboard.writeText(txt); }catch(_){}
      showToast('Copied to clipboard', null);
      return;
    }
  });

  async function loadTrash(){
    if(!trashGrid) return;
    trashGrid.innerHTML = '';
    try{
      const data = await postJson(api('/list_trash.php'), {});
      (data.assets || []).forEach(a => trashGrid.appendChild(makeTrashCard(a)));
      setCounts();
    }catch(e){
      // ignore
    }
  }

  async function refreshLatest(){
    // Load most recent generation assets for user
    try{
      const data = await postJson(api('/list_latest.php'), {});
      adsGrid.innerHTML = '';
      (data.assets || []).forEach(a => adsGrid.appendChild(makeAdCard(a)));
      renderCopy(data.copy || []);
      setCounts();
    }catch(e){}
  }

  $('#emptyTrashBtn')?.addEventListener('click', async ()=>{
    try{
      await postJson(api('/empty_trash.php'), {});
      await loadTrash();
      showToast('Trash emptied', null);
    }catch(e){
      showToast('Error: ' + e.message, null);
    }
  });

  $('#regenCopyBtn')?.addEventListener('click', async ()=>{
    // Re-render from last loaded copy; for real regen call a backend endpoint
    showToast('Tip: regenerate copy by re-running Generate', null);
  });

  $('#generateBtn')?.addEventListener('click', generate);

  // init
  refreshLatest();
  calcCost();
})();