<?php
// app/includes/bunny.php
declare(strict_types=1);

require_once __DIR__ . '/bootstrap.php';

function bunny_cfg(): array {
  static $cfg = null;
  if ($cfg !== null) return $cfg;

  $cfg = require __DIR__ . '/../config/bunny.php';
  if (!is_array($cfg)) $cfg = [];

  $cfg['storage_zone'] = trim((string)($cfg['storage_zone'] ?? ''));
  $cfg['storage_region'] = trim((string)($cfg['storage_region'] ?? ''));
  $cfg['cdn_base_url'] = rtrim((string)($cfg['cdn_base_url'] ?? ''), '/');
  $cfg['base_path'] = trim((string)($cfg['base_path'] ?? ''), '/');

  // Prefer env var for the key (do NOT store in config)
  $cfg['storage_api_key'] = getenv('BUNNY_API_KEY') ?: '';

  if ($cfg['storage_zone'] === '') {
    throw new RuntimeException("Bunny storage_zone missing in config/bunny.php");
  }
  if ($cfg['storage_api_key'] === '') {
    throw new RuntimeException("Bunny API key missing. Set env var BUNNY_API_KEY in /home/adsthatsell/.env");
  }
  if ($cfg['cdn_base_url'] === '') {
    throw new RuntimeException("Bunny cdn_base_url missing in config/bunny.php");
  }

  return $cfg;
}

/**
 * Upload raw bytes to Bunny Storage and return public CDN URL.
 */
function bunny_upload_bytes(string $relativePath, string $bytes, string $contentType = 'application/octet-stream'): string {
  $cfg = bunny_cfg();

  $zone = $cfg['storage_zone'];
  $region = $cfg['storage_region'];
  $key = $cfg['storage_api_key'];

  // Normalize remote path
  $relativePath = ltrim($relativePath, '/');
  if ($cfg['base_path'] !== '') {
    $relativePath = $cfg['base_path'] . '/' . $relativePath;
  }

  // Bunny Storage endpoint format
  // https://{region}.storage.bunnycdn.com/{storageZoneName}/{path}
  $host = ($region !== '') ? "{$region}.storage.bunnycdn.com" : "storage.bunnycdn.com";
  $url = "https://{$host}/{$zone}/{$relativePath}";

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_CUSTOMREQUEST => 'PUT',
    CURLOPT_POSTFIELDS => $bytes,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => [
      "AccessKey: {$key}",
      "Content-Type: {$contentType}",
      "Content-Length: " . strlen($bytes),
    ],
    CURLOPT_TIMEOUT => 120,
  ]);

  $resp = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($resp === false) {
    throw new RuntimeException("Bunny upload failed (curl): {$err}");
  }
  if ($code < 200 || $code >= 300) {
    throw new RuntimeException("Bunny upload failed (HTTP {$code}): " . substr((string)$resp, 0, 400));
  }

  // Public URL (CDN)
  $public = $cfg['cdn_base_url'] . '/' . $relativePath;
  return $public;
}
