<?php
// app/includes/credits.php
declare(strict_types=1);

/**
 * Ensure wallet exists for user.
 */
function ensure_wallet(int $user_id): void {
  $pdo = db();
  $stmt = $pdo->prepare("SELECT id FROM credit_wallets WHERE user_id=? LIMIT 1");
  $stmt->execute([$user_id]);
  if (!$stmt->fetch()) {
    $pdo->prepare("INSERT INTO credit_wallets (user_id,balance,monthly_allowance,allowance_consumed) VALUES (?,?,?,?)")
        ->execute([$user_id, 0, 0, 0]);
  }
}

function wallet_balance(int $user_id): int {
  ensure_wallet($user_id);
  $pdo = db();
  $stmt = $pdo->prepare("SELECT balance FROM credit_wallets WHERE user_id=? LIMIT 1");
  $stmt->execute([$user_id]);
  $row = $stmt->fetch();
  return (int)($row['balance'] ?? 0);
}

/**
 * Atomic debit/credit.
 */
function credit_apply_delta(int $user_id, int $delta, string $reason, string $ref_type='none', ?int $ref_id=null, ?string $memo=null): void {
  $pdo = db();
  ensure_wallet($user_id);

  $pdo->beginTransaction();
  try {
    $bal = wallet_balance_for_update($user_id);
    $new = $bal + $delta;
    if ($new < 0) {
      $pdo->rollBack();
      throw new RuntimeException('INSUFFICIENT_CREDITS');
    }

    $pdo->prepare("INSERT INTO credit_ledger (user_id,delta,reason,ref_type,ref_id,memo,created_at) VALUES (?,?,?,?,?,?,?)")
        ->execute([$user_id, $delta, $reason, $ref_type, $ref_id, $memo, now_utc()]);

    $pdo->prepare("UPDATE credit_wallets SET balance=? WHERE user_id=?")
        ->execute([$new, $user_id]);

    $pdo->commit();
  } catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    throw $e;
  }
}

function wallet_balance_for_update(int $user_id): int {
  $pdo = db();
  $stmt = $pdo->prepare("SELECT balance FROM credit_wallets WHERE user_id=? FOR UPDATE");
  $stmt->execute([$user_id]);
  $row = $stmt->fetch();
  return (int)($row['balance'] ?? 0);
}

/**
 * Compute credit cost for a generation request.
 */
function compute_generation_cost(int $variations, string $quality, bool $generate_copy): int {
  $app = require __DIR__ . '/../config/app.php';
  $mults = $app['credits']['quality_multiplier'] ?? ['medium'=>1.0];
  $m = (float)($mults[$quality] ?? 1.0);
  $cost = (int)max(1, round($variations * $m));
  if ($generate_copy) $cost += (int)($app['credits']['copy_addon'] ?? 1);
  return $cost;
}
