<?php
declare(strict_types=1);

require_once __DIR__ . '/utils.php';

function openai_cfg(): array {
  static $cfg = null;
  if ($cfg !== null) return $cfg;

  $cfg = require __DIR__ . '/../config/openai.php';
  if (!is_array($cfg)) $cfg = [];

  $cfg['api_key'] = (string)($cfg['api_key'] ?? '');
  $cfg['base_url'] = rtrim((string)($cfg['base_url'] ?? 'https://api.openai.com/v1'), '/');
  $cfg['timeout_seconds'] = (int)($cfg['timeout_seconds'] ?? 120);
  $cfg['image_model'] = (string)($cfg['image_model'] ?? 'gpt-image-1.5');
  $cfg['copy_model']  = (string)($cfg['copy_model']  ?? 'gpt-5.2');

  return $cfg;
}

function openai_post_json(string $path, array $payload): array {
  $cfg = openai_cfg();

  $key = $cfg['api_key'] ?: (getenv('OPENAI_API_KEY') ?: '');
  if ($key === '') throw new RuntimeException("OPENAI_API_KEY missing");

  $url = $cfg['base_url'] . $path;

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
      'Authorization: Bearer ' . $key,
      'Content-Type: application/json',
    ],
    CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_SLASHES),
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => $cfg['timeout_seconds'],
  ]);

  $raw = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($raw === false) throw new RuntimeException("OpenAI request failed: $err");

  $json = json_decode($raw, true);
  if (!is_array($json)) throw new RuntimeException("OpenAI returned non-JSON (HTTP $code): " . substr($raw, 0, 300));

  if ($code < 200 || $code >= 300) {
    $msg = $json['error']['message'] ?? ("HTTP $code");
    throw new RuntimeException("OpenAI error: $msg");
  }

  return $json;
}

function http_get_bytes(string $url, int $timeoutSeconds = 60): string {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => $timeoutSeconds,
  ]);
  $raw = curl_exec($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  if ($raw === false || $code < 200 || $code >= 300) return '';
  return (string)$raw;
}

function openai_generate_images(string $prompt, string $size, string $quality, int $n): array {
  $cfg = openai_cfg();

  if (!in_array($size, ['1024x1024','1024x1536','1536x1024'], true)) $size = '1024x1024';
  if (!in_array($quality, ['low','medium','high','auto'], true)) $quality = 'medium';
  $n = max(1, min(8, $n));

  $payload = [
    'model' => $cfg['image_model'],
    'prompt' => $prompt,
    'size' => $size,
    'quality' => $quality,
    'n' => $n,
  ];

  $res = openai_post_json('/images/generations', $payload);

  $out = [];
  $data = $res['data'] ?? [];
  if (!is_array($data)) return $out;

  foreach ($data as $item) {
    if (!is_array($item)) continue;
    if (!empty($item['b64_json']) && is_string($item['b64_json'])) {
      $bin = base64_decode($item['b64_json'], true);
      if ($bin !== false) $out[] = $bin;
      continue;
    }
    if (!empty($item['url']) && is_string($item['url'])) {
      $bin = http_get_bytes($item['url'], $cfg['timeout_seconds']);
      if ($bin !== '') $out[] = $bin;
    }
  }

  return $out;
}

function openai_edit_images(string $prompt, string $image_path, string $size, string $quality, int $n): array {
  $cfg = openai_cfg();
  $key = $cfg['api_key'] ?: (getenv('OPENAI_API_KEY') ?: '');
  if ($key === '') throw new RuntimeException("OPENAI_API_KEY missing");

  if (!is_file($image_path)) throw new RuntimeException("Base image file not found for edits.");
  if (!in_array($size, ['1024x1024','1024x1536','1536x1024'], true)) $size = '1024x1024';
  if (!in_array($quality, ['low','medium','high','auto'], true)) $quality = 'medium';
  $n = max(1, min(8, $n));

  $url = rtrim($cfg['base_url'], '/') . '/images/edits';

  $post = [
    'model' => $cfg['image_model'],
    'prompt' => $prompt,
    'n' => $n,
    'size' => $size,
    'quality' => $quality,
    'image' => new CURLFile($image_path),
  ];

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => $cfg['timeout_seconds'],
    CURLOPT_HTTPHEADER => [
      'Authorization: Bearer ' . $key,
    ],
    CURLOPT_POSTFIELDS => $post,
  ]);

  $raw = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($raw === false) throw new RuntimeException("OPENAI_EDIT_CURL_ERROR: $err");

  $json = json_decode($raw, true);
  if (!is_array($json)) throw new RuntimeException("OPENAI_EDIT_BAD_JSON (HTTP $code): " . substr($raw, 0, 250));
  if ($code < 200 || $code >= 300) {
    $msg = $json['error']['message'] ?? ("HTTP $code");
    throw new RuntimeException("OPENAI_EDIT_ERROR: $msg");
  }

  $out = [];
  foreach (($json['data'] ?? []) as $item) {
    if (!is_array($item)) continue;
    if (!empty($item['b64_json']) && is_string($item['b64_json'])) {
      $bin = base64_decode($item['b64_json'], true);
      if ($bin !== false) $out[] = $bin;
      continue;
    }
    if (!empty($item['url']) && is_string($item['url'])) {
      $bin = http_get_bytes($item['url'], $cfg['timeout_seconds']);
      if ($bin !== '') $out[] = $bin;
    }
  }

  return $out;
}

function openai_generate_copy(string $system, string $user_prompt, string $model): string {
  $cfg = openai_cfg();
  if ($model === '') $model = $cfg['copy_model'];

  $payload = [
    'model' => $model,
    'input' => [
      ['role' => 'system', 'content' => $system],
      ['role' => 'user', 'content' => $user_prompt],
    ],
    'max_output_tokens' => 900,
  ];

  if (!empty($cfg['reasoning_effort'])) {
    $payload['reasoning'] = ['effort' => $cfg['reasoning_effort']];
  }

  $res = openai_post_json('/responses', $payload);

  $text = '';
  if (isset($res['output']) && is_array($res['output'])) {
    foreach ($res['output'] as $o) {
      if (!is_array($o)) continue;
      if (isset($o['content']) && is_array($o['content'])) {
        foreach ($o['content'] as $c) {
          if (is_array($c) && isset($c['text']) && is_string($c['text'])) $text .= $c['text'];
        }
      }
      if (isset($o['text']) && is_string($o['text'])) $text .= $o['text'];
    }
  }
  if ($text === '' && isset($res['output_text']) && is_string($res['output_text'])) $text = $res['output_text'];

  $text = trim($text);
  if ($text === '') throw new RuntimeException("OpenAI Responses returned no text.");
  return $text;
}
