<?php
require __DIR__ . '/includes/bootstrap.php';

if (current_user()) {
  header('Location: ' . app_base() . '/create.php');
  exit;
}

$error = '';
$ok = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_verify_or_die();

  $name  = trim((string)($_POST['name'] ?? ''));
  $email = trim((string)($_POST['email'] ?? ''));
  $pass  = (string)($_POST['password'] ?? '');
  $pass2 = (string)($_POST['password2'] ?? '');

  if ($email === '' || $pass === '') {
    $error = 'Email and password are required.';
  } elseif ($pass !== $pass2) {
    $error = 'Passwords do not match.';
  } elseif (strlen($pass) < 8) {
    $error = 'Password must be at least 8 characters.';
  } else {
    $pdo = db();

    try {
      // Defensive: if some earlier code left a transaction open on this PDO, clear it
      if ($pdo->inTransaction()) {
        $pdo->rollBack();
      }

      // Normalize email (optional but recommended)
      $email_norm = strtolower($email);

      // Make sure we don't race (double submit) — do the uniqueness check inside the transaction
      $pdo->beginTransaction();

      // Lock the email row space (best effort). With a UNIQUE index on users.email this is safe anyway.
      $stmt = $pdo->prepare("SELECT id FROM users WHERE email=? LIMIT 1 FOR UPDATE");
      $stmt->execute([$email_norm]);
      if ($stmt->fetch()) {
        $pdo->rollBack();
        $error = 'Email already exists.';
      } else {
        $hash = password_hash($pass, PASSWORD_DEFAULT);

        $pdo->prepare(
          "INSERT INTO users
            (email, password_hash, name, role, status, default_objective, default_placement, default_copy_type, created_at, updated_at)
           VALUES
            (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        )->execute([
          $email_norm,
          $hash,
          ($name !== '' ? $name : null),
          'user',
          'active',
          'Leads',
          'Facebook Feed',
          'Facebook Ad Copy',
          now_utc(),
          now_utc()
        ]);

        $user_id = (int)$pdo->lastInsertId();

        // IMPORTANT: These functions must NOT start/commit/rollback their own transactions.
        // They should reuse the current transaction. If they do beginTransaction internally,
        // you'll get "There is already an active transaction".
        ensure_wallet($user_id);

        // Signup promo (optional)
        credit_apply_delta($user_id, 50, 'promo', 'none', null, 'Signup bonus');

        $pdo->commit();

        // Session login should happen AFTER commit
        login_user($user_id);

        header('Location: ' . app_base() . '/create.php');
        exit;
      }
    } catch (Throwable $e) {
      if ($pdo->inTransaction()) {
        $pdo->rollBack();
      }
      $error = 'Signup failed: ' . $e->getMessage();
    }
  }
}

$page_title = 'Sign Up';
require __DIR__ . '/includes/header.php';
?>
<header class="topbar">
  <div class="brand">
    <div class="logoMark" aria-hidden="true">
      <svg width="18" height="18" viewBox="0 0 24 24" fill="none">
        <path d="M4 17.5c3.2-7.7 9.1-11.3 16-12.8" stroke="#E10600" stroke-width="2.2" stroke-linecap="round"/>
        <path d="M9 19c2.4-5.2 6.4-8.2 11-9.4" stroke="#fff" stroke-opacity=".9" stroke-width="2.2" stroke-linecap="round"/>
        <path d="M14 20.2c1.2-2.8 3.1-4.4 6-5.2" stroke="#E10600" stroke-width="2.2" stroke-linecap="round"/>
      </svg>
    </div>
    <div class="brandName">AdsThat<span class="sell">Sell</span></div>
  </div>
</header>

<div class="authWrap">
  <form class="authCard" method="post" action="">
    <h1>Create account</h1>
    <p>Start generating ads in minutes.</p>

    <?php if ($error): ?>
      <div class="alert"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">

    <div class="field">
      <div class="label">Name <span style="color:var(--subtle);font-weight:900;">(optional)</span></div>
      <input name="name" type="text" autocomplete="name" value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">
    </div>

    <div class="field">
      <div class="label">Email</div>
      <input name="email" type="email" autocomplete="email" required value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">
    </div>

    <div class="authRow">
      <div class="field">
        <div class="label">Password</div>
        <input name="password" type="password" autocomplete="new-password" required>
      </div>
      <div class="field">
        <div class="label">Confirm</div>
        <input name="password2" type="password" autocomplete="new-password" required>
      </div>
    </div>

    <div class="authActions">
      <button class="btn" type="submit">Create Account</button>
      <a href="<?= htmlspecialchars(app_base()) ?>/login.php">Already have an account?</a>
    </div>
  </form>
</div>

<?php require __DIR__ . '/includes/footer.php'; ?>
